% simulated annealing
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%   Thermodynamically Consistent Model Calibration   %%%%%%%%%%%%
%%%%%%%%%%%              in Chemical Kinetics                  %%%%%%%%%%%%
%%%%%%%%%%%        THERMODYNAMICALLY CONSTRAINED MODEL         %%%%%%%%%%%%
%%%%%%%%%%%                    CALIBRATION                     %%%%%%%%%%%%
%%%%%%%%%%%            Authors: Jenkinson/Goutsias             %%%%%%%%%%%%
%%%%%%%%%%%             Last Modified: 12/02/2010              %%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

function [logKappaFinal,lossKappaFinal,nuFinal] = ...
                   simAnneal(kappa0,B,x0,time,schoeberl,reactToParamMat,dim)
               
%
% initialize parameters
%

phi    = 1.5;    % standard deviation of proposal distribution
S      = 100000; % total number of iterations
U      = 50;     % number of iterations per annealing update
lambda = 2e12;   % initial annealing schedule
delta  = 0.97;	 % decay rate for annealing schedule (<=1)
gamma  = 0.996;  % decay rate for standard deviation of proposal distribution (<=1)

nuFinal     = zeros(size(B,2),1);
lossNuFinal = sqrErrCost(kappa0,schoeberl,reactToParamMat,x0,time,dim);
Cold        = lossNuFinal;
nuOld       = zeros(size(B,2),1);

%
% start annealing
%

wtBar = waitbar(0,'SAN calculations. Please wait ...');

for outerIter = 1:(S/U)
    for innerIter = 1:U
        
        %
        % find new feasible proposal point and its cost function
        %
        
        % randomly choose new nu value
        
        nuPrime = nuOld + phi*randn(size(B,2),1);
        
        % calculate cost at new kappa
        
        Cprime = sqrErrCost(kappa0+B*nuPrime,schoeberl,reactToParamMat,x0,time,dim);
       
        %
        % determine whether moving to new point
        %
        
        if Cprime < Cold   % move to new point for sure 
            
            Cold  = Cprime;
            nuOld = nuPrime;
            
            if Cprime < lossNuFinal  % new point best seen thus far
                lossNuFinal = Cprime;
                nuFinal     = nuPrime;
            end  
            
        elseif rand < exp(-(Cprime-Cold)/lambda) % move to new point with probability  
            Cold  = Cprime;
            nuOld = nuPrime; 
        end
        
    end
    
    %
    % update annealing schedule and standard deviation of proposal
    % distribution
    %
    
    lambda = delta*lambda;
    phi    = gamma*phi;
    waitbar(outerIter/(S/U),wtBar)
    
end

close(wtBar);      

logKappaFinal  = kappa0+B*nuFinal;
lossKappaFinal = lossNuFinal;