% StrSSCplus.m
%% Description: This code is for the Structured Sparse Subspace Clustering (StrSSC or S3C) [1][2]. S3C is implemented in two
% ways: a) hard S3C is presented in [1] where a binary Q is used, and b) soft S3C is presented in [2] where a read-valued Q
% is used. 
% 
%%  StrSSC solves problems as follows:
%
%                     min_{Z,E,Q}  ||Z||_{1,Q} + lambda ||E||
%                     s.t.  D = DZ + E, Diag(Z) = 0,
%                            Q \in \mathcal{Q}       where      
%      -   \mathcal{Q} is the set of all valid segmentation matrix which segment the data set into $k$ groups,
%      -   ||Z||_{1,Q} = ||Z||_1 + \gamma ||Z||_Q = \sum_{i,j} |z_ij|(1 + \gamma \Theta_{ij}) in which 
%          \Theta_{ij} = ||q^i - q^j ||^2_2 / 2 with q^i and q^j being the i-th and j-th row vector of segmentation matrix Q.
%          In soft S3C, the row vector q^i and q^j are normalized to have unit L2 norm.
%      -  ||E|| could be ||E||_1, or ||E||^2_F, or even ||E||_{2,1}.
%
%      We solve this problem by solving two subproblems alternatingly: 
%      Step 1: Given Q, calculate \Theta_{ij} =||q^i - q^j ||^2_2 / 2, and solve for (Z,E). 
%
%                    min_{Z,E}  ||Z||_{1,Q} + lambda ||E||
%                     s.t.  X = XZ + E, Diag(Z) = 0.
%
%                   where  ||Z||_{1,Q} = ||(1 + \Theta ) \odot Z ||_1 in which \odot is element-wise product (i.e. Hadamard
%                   product). This is a problem to find a weighted sparse representation, which can be solved via ADMM.
%
%      Step 2: Given (Z,E), solve for Q. 
%
%                     min_{Q}  ||Z||_{Q}   s.t.  Q \in \mathcal{Q}.  
%
%                    This is a problem to find the segmentation matrix, which can usually be solved via spectral clustering
%                    techniqe by relaxing the binary constraint on Q into column-orthogonal, i.e., Q^T Q = I. 
%                    Here we relax the binary constraint on Q into Q^T D Q = I, where D is the degree matrix of affinity
%                    matrix A = (|Z| +|Z^T|)/2. By doing so, it is effectively equivalent to a normalized cut problem.
%
%%   [acc_i, Theta, Z, eval_iter]  = StrSSCplus(D, idx, opt, DEBUG)
%
%%    Inputs:  D - data matrix, in which each column is a sample
%                 idx - data points groundtruth label
%                 opt - settings of algorithm
%                         opt.sc_method ='lrr'; % e.g., 'ssc', 'StrSSC'
%                         opt.affine = 0 or 1 (or 2 if using L\infty normalization to postprocess C)
%                         opt.outliers =0 or 1
%                         opt.iter_max =10
%                         opt.nu =1, 1.2, etc.
%                         opt.gamma0 =0.2;
%                         opt.lambda =0.3;
%
%                           ADMM parameters:
%                         opt.tol =1e-3;
%                         opt.epsilon =1e-3;
%                         opt.maxIter =1e6;
%                         opt.rho =1.1;
%
%                           Other parameters:
%                         opt.error_norm ='L1', 'L21'   --- type of norm on E
%                         opt.r =0  --- used in SSC or LRR for data preprocessing. If r=0, the orignial data is used;
%                                             otherwise, r-dimensional data obtained by PCA is used.
%                         opt.SSCrho =1   --- used in SSC for postprocessing coefficients (i.e., removing small coefficients).
%                         opt.DEBUG =0   --- used in DEBUG for ADMM.
%
%                 DEBUG - 1 or 0 to turn on / off the middle output, e.g., display matrix C, Theta, etc.
%                 
%%    Outputs: 
%                acc_i  --- accuracy
%                Theta  ---  subspace membership matrix, i.e. the structure indicator matrix
%                Z         ---  sparse representation matrix
%                eval_iter  --- evaluation vs. iteration
%
%%    Reference:
%    [1] Chun-Guang Li and Ren Vidal, "Structured Sparse Subspace Clustering: A Unified 
%          Optimization Framework", in IEEE Conference on Computer Vision and Pattern 
%          Recognition, pp.277-286, June 7-12, 2015. 
%    [2] Chun-Guang Li, Chong You, and Ren Vidal, "Structured Sparse Subspace Clustering: 
%         A Joint Affinity Learning and Subspace Clustering Framework", IEEE Trans. Image 
%         Processing, Vol. 26, No. 6, June 2017, pp.2988-3001.
%
%%    Notes: Several other algorithms, e.g., LRR, LSR, TraceLasso, are also included.
%
% by Chun-Guang Li      
% Date: Jan. 7th, 2014
% Revised: Jan. 21, 2014
% Modified: Oct.31. and Nov. 5,  2014; May 24, 2017
%
function [acc_i, Theta, Z, eval_iter] = StrSSCplus(D, idx, opt, DEBUG)
if nargin < 4
    DEBUG =0;
end
if nargin < 3
    opt.sc_method ='lrr'; % e.g., 'ssc', 'StrSSC'
    opt.affine = 0; % e.g., 0 or 1 for linear subsapce or affine subspace.
    opt.outliers =0; % e.g. 0 or 1: use '||E||^2_F' or '||E||_1'.
    opt.T =1;% T=1 for spectral clustering and T>2 for spectral assumble clustering    
    opt.iter_max =10;
    opt.nu =1;
    opt.gamma0 =0.2;
    opt.lambda =0.3;
    opt.r =0;
    
    opt.tol =1e-3;
    opt.epsilon =1e-3;
    opt.maxIter =1e6;
    opt.rho =1.1;
    opt.error_norm ='L1'; % 'L21'
    opt.DEBUG =0;
    opt.SSCrho =1;
end
epsilon = 1e-8;
T =opt.T;
affine = opt.affine;
outliers =opt.outliers;
nbcluster = max(idx);
grps =0;

switch opt.sc_method
    
    case 'TraceLasso+' % use TraceLasso for Subspace clustering
    
        lambda =opt.lambda;        
        if (affine)  
            D = [ones(1,size(D,2)); D];
            Z = solve_tl(D, lambda); %
        else
            Z =solve_tl(D, lambda);
        end
        Z = Selection( Z , 0.7) ;
        
        if (T < 2)
            [acc_i,~, Theta] =SpectrClustEvaluation(Z, idx, nbcluster, T);
        else
            [acc_i, Theta] = SpectrAssembleClustEvaluation(Z, idx, nbcluster, T);
        end
        
    case 'TraceLasso' % use TraceLasso for Subspace clustering
        lambda =opt.lambda;        
        if (affine)  
            D = [ones(1,size(D,2)); D];
            Z = solve_tl(D, lambda); %
        else
            Z =solve_tl(D, lambda, opt.tol);
        end
        
        if (T < 2)
            [acc_i,~, Theta] =SpectrClustEvaluation(Z, idx, nbcluster, T);
        else
            [acc_i, Theta] = SpectrAssembleClustEvaluation(Z, idx, nbcluster, T);
        end    
    
    case 'LSR1' % use LSR1 for Subspace clustering
        lambda =opt.lambda;        
        if (affine)  
            D = [ones(1,size(D,2)); D];
            Z = LSR1(D, lambda, opt); %
        else
            Z =LSR1(D, lambda);
        end
        
        if (T < 2)
            [acc_i,~, Theta] =SpectrClustEvaluation(Z, idx, nbcluster, T);
        else
            [acc_i, Theta] = SpectrAssembleClustEvaluation(Z, idx, nbcluster, T);
        end
        
    case 'LSR2' % use LSR2 for Subspace clustering
        lambda =opt.lambda;        
        if (affine)  
            D = [ones(1,size(D,2)); D];
            Z = LSR2(D, lambda, opt); %% ???
        else
            Z =LSR2(D, lambda);
        end
        
        if (T < 2)
            [acc_i,~, Theta] =SpectrClustEvaluation(Z, idx, nbcluster, T);
        else
            [acc_i, Theta] = SpectrAssembleClustEvaluation(Z, idx, nbcluster, T);
        end
        
    case 'LADM-lrr' % use LRR for Subspace clustering
        lambda =opt.lambda;        
        if (affine)  
            D = [ones(1,size(D,2)); D];
            Z = lrr_aff(D, lambda, opt);
        else
            Z =lrr_ladm(D, lambda, opt);
        end
        
        if (T < 2)
            [acc_i,~, Theta] =SpectrClustEvaluation(Z, idx, nbcluster, T);
        else
            [acc_i, Theta] = SpectrAssembleClustEvaluation(Z, idx, nbcluster, T);
        end        
        
    
    case 'lrr' % use LRR for Subspace clustering
        lambda =opt.lambda;        
        if (affine)  
            D = [ones(1,size(D,2)); D];
            Z = lrr_aff(D, lambda, opt);
        else
            Z =lrr(D, lambda, opt);
        end
        
        if (T < 2)
            [acc_i,~, Theta] =SpectrClustEvaluation(Z, idx, nbcluster, T);
        else
            [acc_i, Theta] = SpectrAssembleClustEvaluation(Z, idx, nbcluster, T);
        end
        
        % the hard S3C:
    case {'StrSSC'; 'StrSSC-Fix'; 'StrSSC-Up';} % use mixed norm, StrLRR for Subspace clustering
        
        gamma0 =opt.gamma0;
        opt.Z =zeros(size(D,2));
        lambda0 =opt.lambda;        
        r = opt.r; % r: the dimension of the target space when applying PCA or random projection
        Xp = DataProjection(D,r);
        %Theta_old =ones(size(X, 2)); % if we not initialize gamma1=0, we should initialize Theta_old =ones.       
        Theta_old =ones(size(D, 2));
        iter_max =opt.iter_max;%iter_max =10; for SSC and Spectral Clustering
        maxIter =opt.maxIter; % in ADMM
        nu =opt.nu; % nu=1;
        rho =opt.SSCrho; %rho = 0.7;
        Z =zeros(size(Xp,2));
        iter =0; 
        %Centriod =0;
        eval_iter =zeros(8, iter_max);                
        old_obj =Inf;
        %old_Z_Q_norm =Inf;
        %odd_iter =0;
        while (iter < iter_max)
            
            iter = iter +1;
            
           switch opt.sc_method
               
                case {'StrSSC';}
                    if (iter <= 1)
                        %% This is the standard SSC: lrr or lrr_affine
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC: SVT + Soft thresholding on Z
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
%                     nu = nu * (1.2)^(iter - 1);%1.1, 1.2;%1.5, 1.2
%                     gamma1 =gamma0;
                    %% 
                    [C, E] = ADMM_StrSSC(Xp, outliers, affine, lambda0, Theta_old, gamma1, nu,  maxIter, Z);      
                    
                case  {'StrSSC-Fix';}
                    if (iter <= 1)
                        %% This is the standard SSC
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.0;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_StrSSC(Xp, outliers, affine, lambda0, Theta_old, gamma1, nu,  maxIter, Z);                       
                    
                case  {'StrSSC-Up';}
                    if (iter <= 1)
                        %% This is the standard SSC
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_StrSSC_Up(Xp, outliers, affine, lambda0, Theta_old, gamma1, nu,  maxIter, Z);
            end            
          
            %% Initialize Z with the last optimal Z.
            Z =C;
            
            if (DEBUG)
                figure;
                image(abs(C)*800);colorbar;
            end
            
            CKSym = BuildAdjacency(thrC(C,rho)); % 
            if (DEBUG)
                figure;
                image(abs(CKSym)*800);colorbar;
            end
            
            %% Need to take care of this line. The last input 'grps' is used for a warmstart of k-means in the next call, in
            %% which the standard k-means is slightly changed to allow for this extra input. 
            [grps] = SpectralClustering(CKSym, nbcluster, grps);
            %% If you want to use the k-means in default, please use: 
            % grps = SpectralClustering(CKSym, nbcluster);
            
            if (nbcluster >10)
                missrate = 1 - evalAccuracyHungarian(grps, idx');
            else
                missrate = Misclassification(grps, idx');
            end
            disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate)]);
            acc_i =1 - missrate;

            av_conn = evalAverConn(0.5*(abs(C)+abs(C')), idx'); 
            [ssr_err, ssr ]  = evalSSR_error(C, idx');  
            eval_iter(1, iter) = 1 - missrate;
            eval_iter(2, iter) = av_conn;                
            eval_iter(3, iter) = ssr_err;
            eval_iter(4, iter) = ssr;
            %eval_iter(4, iter) = evalSSR_perc(C, idx');     

            Theta = form_structure_matrix(grps);                 
            
            if (DEBUG)
                figure;
                image(abs(1-Theta)*800);colorbar;
            end
            
            if (outliers)
                lambda = lambda0 / norm(Xp,1);
                obj = sum(sum(abs((1 + gamma0*Theta).*C))) + lambda*sum(sum(E)); 
            else
                %%
                lambda = lambda0 / norm(Xp,1);
                obj = sum(sum(abs((1 + gamma0*Theta).*C))) + 0.5*lambda* trace((Xp - Xp*C)*(Xp-Xp*C)'); 
            end
            
            %% Relative Objective:
            if (iter ==1)
                relative_obj =1;
                eval_iter(8, iter) = relative_obj;
            else
                relative_obj =(obj - old_obj) /(old_obj + (old_obj < eps));
                eval_iter(8, iter) = relative_obj;
            end
            
            Z_Q_norm = sum(sum(abs((Theta).*C)));
            disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);
            
            eval_iter(5, iter) = obj;
            eval_iter(6, iter) = Z_Q_norm;       

            
            %% Checking stop criterion
            tmp =Theta - Theta_old;
            
            %if (max(abs(tmp(:))) < 0.5) && (max(max((1-Theta).*Z)) < 5e-3)
            if (max(abs(tmp(:))) < 0.5)
                break; % if Q didn't change, stop the iterations.
            end
            
            Theta_old =Theta;
            %gamma0 = gamma0*1.1; %1.1 makes nS=15 works; 1.5 makes nS=5 works. 1.2; % 1.5
        end
                
        
        %% A modification of hard StrSSC
    case {'StrSSC-Fix2';'StrSSC-Up2';'StrSSC2';'StrSSC-Fix2+';'StrSSC-Up2+';'StrSSC2+';} %  hard StrSSC for Subspace clustering
       
        gamma0 =opt.gamma0;
        opt.Z =zeros(size(D,2));
        lambda0 =opt.lambda;        
        r = opt.r; % r: the dimension of the target space when applying PCA or random projection
        Xp = DataProjection(D,r);
        %Theta_old =ones(size(X, 2)); % if we not initialize gamma1=0, we should initialize Theta_old =ones.       
        Theta_old =ones(size(D, 2));        
        %softTheta =zeros(size(D, 2));
        softTheta_old =1;
        %softTheta =zeros(size(D, 2));
        iter_max =opt.iter_max;%iter_max =10; for SSC and Spectral Clustering
        maxIter =opt.maxIter; % in ADMM
        nu =opt.nu; % nu=1;
        rho =opt.SSCrho; %rho = 0.7;
        Z =zeros(size(Xp,2));
        
        eval_iter =zeros(13, iter_max);      
        old_obj =Inf;       
        m_Z_Q_norm_old = Inf;
        
        % m_rel_C_old = Inf;
        % m_rel_softTheta_old = Inf;
        % m_cost_kmeans_old = Inf;      
        
        iter =0; odd_iter =0;
        while (iter < iter_max)
            
            iter = iter +1;            
            switch opt.sc_method
                
                case {'StrSSC2';'StrSSC2+'}
                    if (iter <= 1)
                        %% This is the standard SSC
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_StrSSC(Xp, outliers, affine, lambda0, Theta_old, gamma1, nu,  maxIter, Z);      
                    
                case  {'StrSSC-Fix2'; 'StrSSC-Fix2+';}
                    if (iter <= 1)
                        %% This is the standard SSC
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.0;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_StrSSC(Xp, outliers, affine, lambda0, Theta_old, gamma1, nu,  maxIter, Z);        
                    
                case  {'StrSSC-Up2'; 'StrSSC-Up2+'}
                    if (iter <= 1)
                        %% This is the standard SSC
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_StrSSC_Up(Xp, outliers, affine, lambda0, Theta_old, gamma1, nu,  maxIter, Z); 
            end
            
            % relative changes in C
            tmp = Z - C;
            eval_iter(9, iter) = sum(abs(tmp(:))) / (sum(abs(C(:))) + (sum(abs(C(:))) < 1e-6));
            %% Initialize Z with the last optimal Z.
            Z = C;
            
            if (DEBUG)
                figure;
                image(abs(C)*800);colorbar;
            end
                           
            if (affine==2)  % affine==2 : using L1 normalization. If rho <1 : filtering coefficients
                CKSym = BuildAdjacency(thrC(C,rho)); % L1 normalization + filtering
            else if (rho <1.0)
                    CKSym = thrC(C,rho); % e.g.,rho=0.7, it is filtering coefficients without L1 normalization
                    CKSym = 0.5* (abs(CKSym) + abs(CKSym'));
                else
                    CKSym = 0.5* (abs(C) + abs(C')); % no any postprocessing on C
                end
            end
            
%             if (DEBUG)
%                 figure;
%                 image(abs(CKSym)*800);colorbar;
%             end

            switch opt.sc_method                    
                case {'StrSSC2'; 'StrSSC-Fix2'; 'StrSSC-Up2';}
                    [grps, ~, ~, Q, dim_kerN, num_spaU, sumD] = mSpectralClustering(CKSym, nbcluster);%grps           

                case {'StrSSC2+'; 'StrSSC-Fix2+'; 'StrSSC-Up2+';}
                    [grps, ~, ~, Q, dim_kerN, num_spaU, sumD] = mSpectralClustering(CKSym, nbcluster, grps);%grps           
            end
            
            if (nbcluster >10)
                missrate = 1 - evalAccuracyHungarian(grps, idx');
            else
                missrate = Misclassification(grps, idx');
            end                               

            av_conn = evalAverConn(0.5*(abs(C)+abs(C')), idx'); 
            [ssr_err, ssr ]  = evalSSR_error(C, idx');
            
            softTheta = L2_distance(Q', Q');
            softTheta = (softTheta.^2) / 2;                
            %Theta = squareform(pdist(Q) .^2) / 2;
            
            tmp = softTheta - softTheta_old;  
            eval_iter(11, iter) = sum( abs(tmp(:)) ) / sum( abs(softTheta(:)));
            softTheta_old =softTheta;
 
            eval_iter(12, iter) =min(sumD(:)); % / (size(C, 1) * nbcluster);
            eval_iter(13, iter) =mean(sumD(:)); % / (size(C, 1) * nbcluster);

            Theta = form_structure_matrix(grps);

            %affine = opt.affine;
            %outliers =opt.outliers;
            if (outliers)
                lambda = lambda0 / norm(Xp,1);
                obj = sum(sum(abs((1 + gamma0*(1-Theta)).*C))) + lambda*sum(sum(E)); 
            else
                %%                     
                lambda = lambda0 / norm(Xp,1);
                obj = sum(sum(abs((1 + gamma0*(1-Theta)).*C))) + 0.5*lambda* trace((Xp - Xp*C)*(Xp-Xp*C)'); 
            end

            %% Relative Objective:
            if (iter ==1)
                relative_obj =1;
                eval_iter(8, iter) = relative_obj;
            else
                relative_obj =(obj - old_obj) /(old_obj + (old_obj < eps));
                eval_iter(8, iter) = relative_obj;
            end                
            
            if (0==1)
                %figure;imagesc(softTheta, [0, 1]);title('soft \Theta'); colorbar;
                figure;imagesc(1-Theta, [0, 1]);colorbar;title('hard \Theta');
            end
            if (DEBUG)
                figure;
                image(abs(1-Theta)*800);colorbar;
            end
            
            Z_Q_norm = sum(sum(abs((1-Theta).*C)));
            
            %% Checking stop criterion
            tmp =Theta - Theta_old;            
            eval_iter(10, iter) = sum( abs(tmp(:)) ) / sum( abs(Theta(:)));
            %disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);
   
            disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);                
            disp(['iter = ',num2str(iter),', rel_Z = ', num2str(eval_iter(9, iter)), ', rel_Theta =',num2str(eval_iter(10, iter)),',  rel_softTheta=', num2str(eval_iter(11, iter)), ', sumD = ', num2str(eval_iter(12, iter)),  ', mean sumD = ', num2str(eval_iter(13, iter))]);
                        
            eval_iter(1, iter) = 1 - missrate;
            eval_iter(2, iter) = av_conn;                
            eval_iter(3, iter) = ssr_err;
            %eval_iter(4, iter) = evalSSR_perc(C, idx');                
            eval_iter(4, iter) = obj;            
            eval_iter(5, iter) = Z_Q_norm;       

            eval_iter(6, iter) = dim_kerN;                
            eval_iter(7, iter) = num_spaU;             
            %%%%%%%%%%%%%%%%%%           

            %% Stopping rules:
            m_Z_Q_norm =eval_iter(5, iter);
            m_rel_Theta =eval_iter(10, iter);
            
            if (num_spaU >0.9) || (dim_kerN >= nbcluster)                  
                odd_iter = odd_iter +1;
                if (m_rel_Theta < epsilon) || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon)) || (odd_iter >3)
                    break; 
                end
            else
                if (m_rel_Theta < epsilon) || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon))
                    break; % if Q didn't change, stop the iterations.
                end
            end
            acc_i =1 - missrate;
            
%%           %%%%%%%%%%%%%%%%% An alternative stopping rule:            
%             m_rel_C =eval_iter(9, iter);
%             m_rel_softTheta =eval_iter(11, iter);
%             m_cost_kmeans =eval_iter(13, iter);            
%             if (num_spaU > 0)   
%                 acc_i =1 - missrate;
%                 break;                
%             else
%                 % if (m_rel_Theta <1e-8) || (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                 if (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || ...
%                         (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                     break;
%                 end
%             end
%             acc_i =1 - missrate;
%             m_rel_C_old = m_rel_C;
%             m_rel_softTheta_old = m_rel_softTheta;
%             m_cost_kmeans_old = m_cost_kmeans;            
%             %%%%%%%%%%%%%%%%%
            
            old_obj = obj;
            Theta_old =Theta;
            m_Z_Q_norm_old = m_Z_Q_norm;
        end             
        
    case {'softStrSSC-Fix2';  'softStrSSC-Fix2+';   'softStrSSC-Up2'; 'softStrSSC-Up2+';...
            'softStrSSC2'; 'softStrSSC2+'; 'softStrSSC2+un'} % soft StrSSC for Subspace clustering
       
        gamma0 =opt.gamma0;
        opt.Z =zeros(size(D,2));
        lambda0 =opt.lambda;        
        r = opt.r; % r: the dimension of the target space when applying PCA or random projection
        Xp = DataProjection(D,r);
        %Theta_old =ones(size(X, 2)); % if we not initialize gamma1=0, we should initialize Theta_old =ones.       
        Theta_old =ones(size(D, 2));
        softTheta =zeros(size(D, 2));
        softTheta_old =1;
        iter_max =opt.iter_max;%iter_max =10; for SSC and Spectral Clustering
        maxIter =opt.maxIter; % in ADMM
        nu =opt.nu; % nu=1;
        rho =opt.SSCrho; %rho = 0.7;
        Z =zeros(size(Xp,2));
        
        eval_iter =zeros(13, iter_max);      
        old_obj =Inf;
        m_Z_Q_norm_old = Inf;
        
%         m_rel_C_old = Inf;
%         m_rel_softTheta_old = Inf;
%         m_cost_kmeans_old = Inf;
%         k_arr =[];        
        
        iter =0; odd_iter =0;        
        while (iter < iter_max)            
            iter = iter +1;            
            switch opt.sc_method                
                %% 
                case {'softStrSSC-Fix2';'softStrSSC-Fix2+'}
                    if (iter <= 1)
                        %% This is the standard SSC
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.0;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_softStrSSC(Xp, outliers, affine, lambda0, softTheta, gamma1, nu,  maxIter, Z);             
                    
                 %%
                case {'softStrSSC-Up2';'softStrSSC-Up2+';}
                    if (iter <= 1)
                        %% This is the standard SSC
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_softStrSSC_Up(Xp, outliers, affine, lambda0, softTheta, gamma1, nu,  maxIter, Z);                    
                    
                case {'softStrSSC2';'softStrSSC2+';}
                    if (iter <= 1)
                        %% This is the standard SSC
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %%
                    [C, E] = ADMM_softStrSSC(Xp, outliers, affine, lambda0, softTheta, gamma1, nu,  maxIter, Z);                    
                     
            end                            
            
%             %% TEST for detecting k when k is unknown
%             if strcmp(opt.sc_method, 'softStrSSC2+un')
%                 tmpC = 0.5 * (abs(C) + abs(C'));            
%                 %%  Unnormalized Laplacian
%                 Lap = diag(sum(tmpC)) - tmpC;            
%                 %% SVD
%                 [~,s,~] = svd(Lap);
%                 s =diag(s);
%                 [K, P1, P2, P3, P4] =detect_num_subspaces(s);
%                 disp(['K: ', num2str(K)]);
%                 nbcluster = K(2);
%                 k_arr =[k_arr; nbcluster];
%                 if (nbcluster ==max(idx))
%                     disp('correct!!');
%                 else
%                     disp('Wrong! *********************')
%                 end
%                 %show_detect_k_result;
%             end
            
            % relative changes in C
            tmp = Z - C;
            eval_iter(9, iter) = sum(abs(tmp(:))) / (sum(abs(C(:))) + (sum(abs(C(:))) < 1e-6));
            
            %% Initialize Z with the last optimal Z.
            Z =C;       
            
            if (DEBUG)
                figure;
                colormap hot;
                image(abs(C)*800);colorbar;
                figure;
                colormap hot;
                image(abs(BuildAdjacency(thrC(C,rho)))*800);colorbar;
            end
            
            if (affine==2)  % affine==2 : using L1 normalization. If rho <1 : filtering coefficients
                CKSym = BuildAdjacency(thrC(C,rho)); % L1 normalization + filtering
            else if (rho <1.0)
                    CKSym = thrC(C,rho); % e.g.,rho=0.7, it is used to filter the coefficients (without L1 normalization)
                    CKSym = 0.5* (abs(CKSym) + abs(CKSym'));
                else
                    CKSym = 0.5* (abs(C) + abs(C')); % if affine=0 or 1, and rho=1, then there is no any postprocessing on C
                end
            end
            
%             if (DEBUG)
%                 figure;
%                 image(abs(CKSym)*800);colorbar;
%             end
            
            if max(strcmp(opt.sc_method, {'softStrSSC2+';'softStrSSC-Fix2+';'softStrSSC-Up2+'}))
                [grps, ~,~, Q, dim_kerN, num_spaU, sumD] = mSpectralClustering(CKSym, nbcluster, grps);% with warmstart
            else
                [grps, ~,~, Q, dim_kerN, num_spaU, sumD] = mSpectralClustering(CKSym, nbcluster);% without warmstart
            end
            
            softTheta = L2_distance(Q', Q');
            softTheta = (softTheta.^2) / 2;
            %Theta = squareform(pdist(Q) .^2) / 2;            
            
            Theta = form_structure_matrix(grps);
            
            %affine = opt.affine;
            %outliers =opt.outliers;
            if (outliers)
                lambda = lambda0 / norm(Xp,1);
                obj = sum(sum(abs((1 + gamma0*softTheta).*C))) + lambda * sum(sum(E)); 
            else
                %%                     
                lambda = lambda0 / norm(Xp,1);
                obj = sum(sum(abs((1 + gamma0*softTheta).*C))) + 0.5 * lambda * trace((Xp - Xp*C)*(Xp-Xp*C)'); 
            end
            
            if (nbcluster >10)
                %missrate = 1 - compacc(grps, idx');
                missrate = 1 - evalAccuracyHungarian(grps, idx');
            else
                missrate = Misclassification(grps, idx');
            end                               

            av_conn = evalAverConn(0.5*(abs(C)+abs(C')), idx'); 
            [ssr_err, ssr ]  = evalSSR_error(C, idx');  
            
            eval_iter(1, iter) = 1 - missrate;
            eval_iter(2, iter) = av_conn; 
            eval_iter(3, iter) = ssr_err;
            %eval_iter(4, iter) = evalSSR_perc(C, idx');                
            eval_iter(4, iter) = obj;            
            
            
            Z_Q_norm = sum(sum(abs((softTheta).*C)));
            eval_iter(5, iter) = Z_Q_norm;       

            eval_iter(6, iter) = dim_kerN;                
            eval_iter(7, iter) = num_spaU;          
            
            %% Relative Objective:
            if (iter ==1)
                acc_i =1 - missrate;
                relative_obj =1;
                eval_iter(8, iter) = relative_obj;
            else
                relative_obj =(obj - old_obj) /(old_obj + (old_obj < eps));
                eval_iter(8, iter) = relative_obj;
            end                 

            if (0==1)
                figure;imagesc(softTheta, [0, 1]);title('soft \Theta'); colorbar;
                figure;imagesc(1-Theta, [0, 1]);colorbar;title('hard \Theta');
            end
            if (DEBUG)
                figure;colormap hot
                image(abs(1-Theta)*800);colorbar;                
                figure;colormap hot
                imagesc(softTheta, [0, 1]);title('soft \Theta'); colorbar;
                figure;colormap hot
                image(softTheta*500);title('soft \Theta'); colorbar;
            end
            
            %% Checking stopping condition
            tmp = softTheta - softTheta_old;
            eval_iter(11, iter) = sum( abs(tmp(:)) ) / sum( abs(softTheta(:)));
            softTheta_old =softTheta;
            
            tmp =Theta - Theta_old;            
            eval_iter(10, iter) = sum( abs(tmp(:)) ) / sum( abs(Theta(:)));

            eval_iter(12, iter) =min(sumD(:)); % / (size(C, 1) * nbcluster);
            eval_iter(13, iter) =mean(sumD(:)); % / (size(C, 1) * nbcluster);                        
            
            disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);                
            disp(['iter = ',num2str(iter),', rel_Z = ', num2str(eval_iter(9, iter)), ', rel_Theta =',num2str(eval_iter(10, iter)),',  rel_softTheta=', num2str(eval_iter(11, iter)), ', sumD = ', num2str(eval_iter(12, iter)),  ', mean sumD = ', num2str(eval_iter(13, iter))]);
                     
            %% Stopping rules:
            m_Z_Q_norm =eval_iter(5, iter);
            m_rel_Theta =eval_iter(10, iter);
            
            if (num_spaU >0.9) || (dim_kerN >= nbcluster)                  
                odd_iter = odd_iter +1;
                if (m_rel_Theta < epsilon) || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon)) || (odd_iter >3)
                    break; 
                end
            else
                if (m_rel_Theta < epsilon) || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon))
                    break; % if Q didn't change, stop the iterations.
                end
            end
            acc_i =1 - missrate;
            
%%           %%%%%%%%%%%%%%%%% An alternative stopping rule:            
%             m_rel_C =eval_iter(9, iter);
%             m_rel_softTheta =eval_iter(11, iter);
%             m_cost_kmeans =eval_iter(13, iter);            
%             if (num_spaU > 0)   
%                 acc_i =1 - missrate;
%                 break;                
%             else
%                 % if (m_rel_Theta <1e-8) || (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                 if (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || ...
%                         (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                     break;
%                 end
%             end
%             acc_i =1 - missrate;
%             m_rel_C_old = m_rel_C;
%             m_rel_softTheta_old = m_rel_softTheta;
%             m_cost_kmeans_old = m_cost_kmeans;            
%             %%%%%%%%%%%%%%%%%

            m_Z_Q_norm_old = m_Z_Q_norm;
            old_obj = obj;
            Theta_old =Theta;
            
        end             
                
    otherwise        
        disp('Unkown Subspace Clustering Methods...');        
end

function M = form_structure_matrix(idx,n)
if nargin<2
    n =size(idx,2);
end
M =zeros(n);
id =unique(idx);
for i =1:length(id)
    idx_i =find(idx == id(i));
    M(idx_i,idx_i)=ones(size(idx_i,2));
end  

% function Dist = form_distance_matrix(X,t)
% if nargin<2
%     t =1;
% end
% 
% for i =1:length(id)
%     idx_i =find(idx == id(i));
%     M(idx_i,idx_i)=ones(size(idx_i,2));
% end  