% ConstraintStrSSC.m
% Description: This code is for the Constrained Structured Sparse Subspace Clustering (CStrSSC or CS3C) [2]. CS3C is
% implemented upon two versions of S3C [1][2]: a) Constraint hard S3C (where a binary Q is used in S3C [1]), and b)
% Constraint soft S3C (where a read-valued Q is used in S3C [2]). 
% 
%%  Constraint StrSSC:
%                     min_{Z,E,Q}  ||Z||_{\Psi,Q} + lambda ||E||
%                     s.t.  D = DZ + E, Diag(Z) = 0, Q \in \mathcal{Q},       
%      where
%      -   \Psi is a matrix of side-information to provide the constraints in subspace clustering. If there is no
%      side-information, \Psi is set as 1 (i.e. ones(N,N), a matrix of all 1's.)
%      -   \mathcal{Q} is the set of all valid segmentation matrix which segment the data set into $k$ groups,
%      -   ||Z||_{\Psi,Q} = ||Z \odot \Psi ||_1 + \gamma ||Z||_Q = \sum_{i,j} |z_ij|(\Psi_{ij} + \gamma \Theta_{ij}) in which 
%          \Theta_{ij} = ||q^i - q^j ||^2_2 / 2 with q^i and q^j being the i-th and j-th row vector of segmentation matrix Q.
%          In soft S3C, the row vector q^i and q^j are normalized to have unit L2 norm.
%      -  ||E|| could be ||E||_1, or ||E||^2_F, or even ||E||_{2,1}.
%
%      We solve this problem by solving two subproblems alternatingly: 
%      Step 1: Given Q, calculate \Theta_{ij} =||q^i - q^j ||^2_2 / 2, and solve for (Z,E). 
%
%                    min_{Z,E}  ||Z||_{\Psi,Q} + lambda ||E||
%                     s.t.  X = XZ + E, Diag(Z) = 0.
%
%                   where  ||Z||_{\Psi,Q} = ||(\Psi +\gamma \Theta ) \odot Z ||_1 in which \odot is element-wise product (i.e. Hadamard
%                   product). This is a problem to find a weighted sparse representation, which can be solved via ADMM.  Note
%                   that the term \Psi can be combined with \Theta and thus we can solve the problem with minor changes via
%                   the S3C algorithms.
%
%      Step 2: Given (Z,E), solve for Q. 
%
%                     min_{Q}  ||Z||_{Q}   s.t.  Q \in \mathcal{Q}.  
%
%                    This is a problem to find the segmentation matrix, which can usually be solved via spectral clustering
%                    techniqe by relaxing the binary constraint on Q into column-orthogonal, i.e., Q^T Q = I. 
%                    Here we relax the binary constraint on Q into Q^T D Q = I, where D is the degree matrix of affinity
%                    matrix A = (|Z| +|Z^T|)/2. By doing so, it is effectively equivalent to a normalized cut problem.
%
%%   [acc_i, Theta, Z, eval_iter]  = ConstraintStrSSC(D, idx, opt, DEBUG)
%
%%    Inputs:  D - data matrix, in which each column is a sample
%                 idx - data points groundtruth label
%                 opt - settings of algorithm
%                         opt.Psi : side-information matrix. If Psi =1 (i.e. ones(N)), it reduces to S3C. 
%                         opt.sc_method ='lrr'; % e.g., 'ssc', 'StrSSC'
%                         opt.affine = 0 or 1 (or 2 if using L\infty normalization to postprocess C)
%                         opt.outliers =0 or 1
%                         opt.iter_max =10
%                         opt.nu =1, 1.2, etc.
%                         opt.gamma0 =0.2;
%                         opt.lambda =0.3;
%
%                           ADMM parameters:
%                         opt.tol =1e-3;
%                         opt.epsilon =1e-3;
%                         opt.maxIter =1e6;
%                         opt.rho =1.1;
%
%                           Other parameters:
%                         opt.error_norm ='L1', 'L21'   --- type of norm on E
%                         opt.r =0  --- used in SSC or LRR for data preprocessing. If r=0, the orignial data is used;
%                                             otherwise, r-dimensional data obtained by PCA is used.
%                         opt.SSCrho =1   --- used in SSC for postprocessing coefficients (i.e., removing small coefficients).
%                         opt.DEBUG =0   --- used in DEBUG for ADMM.
%
%                 DEBUG - 1 or 0 to turn on / off the middle output, e.g., display matrix C, Theta, etc.
%                 
%%    Outputs: 
%                acc_i  --- accuracy
%                Theta  ---  subspace membership matrix, i.e. the structure indicator matrix
%                Z         ---  sparse representation matrix
%                eval_iter  --- evaluation vs. iteration
%                CONS  --- the Rand Index Estimator, which is used to perform parameter selection
%
%%    Reference:
%    [1] Chun-Guang Li and Ren Vidal, "Structured Sparse Subspace Clustering: A Unified 
%          Optimization Framework", in IEEE Conference on Computer Vision and Pattern 
%          Recognition, pp.277-286, June 7-12, 2015. 
%    [2] Chun-Guang Li, Chong You, and Ren Vidal, "Structured Sparse Subspace Clustering: 
%         A Joint Affinity Learning and Subspace Clustering Framework", IEEE Trans. Image 
%         Processing, Vol. 26, No. 6, June 2017, pp.2988-3001.
% 
% Copyright by Chun-Guang Li      
% Apr. 8,  2014; Modified: July 14, 2017
function [acc_i, Theta, Z, eval_iter, CONS] = ConstraintStrSSC(D, idx, opt, DEBUG)
if nargin < 4
    DEBUG =0;
end
if nargin < 3
    opt.Psi =1;
    opt.sc_method ='lrr'; % e.g., 'ssc', 'StrSSC'
    opt.affine = 0; % e.g., 'ssc'
    opt.outliers =0;
    opt.T =1;% T=1 for spectral clustering and T>2 for spectral assumble clustering    
    opt.iter_max =10;
    opt.nu =1;
    opt.gamma0 =0.2;
    opt.lambda =0.3;
    opt.r =0;    
    opt.tol =1e-3;
    opt.epsilon =1e-3;
    opt.maxIter =1e6;
    opt.rho =1.1;
    opt.error_norm ='L1'; % 'L21'
    opt.DEBUG =0;
    opt.SSCrho =1;
end
epsilon = 1e-8;
%T =opt.T;
affine = opt.affine;
outliers =opt.outliers;
nbcluster = max(idx);
grps =0;

% if isfield(opt,'t')
%     t = opt.t;
% else
%     t =0;
% end

switch opt.sc_method      
        %% 1.  StrSSC with initialization from "cannot-link" constraints
        
    case {'StrSSC_init2';'StrSSC_init-Fix2';'CStrSSC_init-Up2'; ...
            'StrSSC_init2+';'StrSSC_init-Fix2+';'CStrSSC_init-Up2+';...
            'softStrSSC_init2';'softStrSSC_init-Fix2'; 'softStrSSC_init-Up2';...
            'softStrSSC_init2+';'softStrSSC_init-Fix2+'; 'softStrSSC_init-Up2+';} 
       
        gamma0 =opt.gamma0;
        opt.Z =zeros(size(D,2));
        alpha =opt.lambda;        
        r = opt.r; % r: the dimension of the target space when applying PCA or random projection
        Xp = DataProjection(D,r);
        
        iter_max =opt.iter_max;%iter_max =10; for SSC and Spectral Clustering
        maxIter =opt.maxIter; % in ADMM
        nu =opt.nu; % nu=1;
        rho =opt.SSCrho; %rho = 0.7;
        Z =zeros(size(Xp,2));
        
        %% given prior side-information
        Psi =opt.Psi;
        Theta_old =zeros(size(D, 2));
        Theta_old((Psi >=0)) = Psi(Psi >=0);
        Theta_old = 1 -Theta_old;
        
        eval_iter =zeros(13, iter_max);      
        old_obj =Inf;       
        m_Z_Q_norm_old = Inf;
        
        iter =0;  odd_iter =0;
        while (iter < iter_max)
            
            iter = iter +1;            
            gamma1 =gamma0;            
            switch opt.sc_method   
                
                case {'StrSSC_init2'; 'StrSSC_init2+'}
                    if (iter <= 1)
                        %% This is the standard SSC with initialized Theta
                        nu =1; 
                    else
                        %% This is the re-weighted SSC: with initialized Theta
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                    end
                    %% 
                    [C, E] = ADMM_StrSSC(Xp, outliers, affine, alpha, Theta_old, gamma1, nu,  maxIter, Z);      
                    
                case  {'StrSSC_init-Fix2'; 'StrSSC_init-Fix2+'}
                    nu =1; 
                    %% 
                    [C, E] = ADMM_StrSSC(Xp, outliers, affine, alpha, Theta_old, gamma1, nu,  maxIter, Z);                       
                    
                case  {'StrSSC_init-Up2'; 'StrSSC_init-Up2+'}
                    if (iter <= 1)
                        nu =1; 
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                    end
                    %% 
                    [C, E] = ADMM_StrSSC_Up(Xp, outliers, affine, alpha, Theta_old, gamma1, nu,  maxIter, Z);                      
             
                %% 
                case {'softStrSSC_init-Fix2'; 'softStrSSC_init-Fix2+'}
                    nu =1;                    
                    %% 
                    %C = ADMM_StrSSC(Xp, outliers, affine, alpha, Theta_old, gamma1, nu,  maxIter, Z);
                    [C, E] = ADMM_softStrSSC(Xp, outliers, affine, alpha, softTheta, gamma1, nu,  maxIter, Z);             
                    
                 %%
                case {'softStrSSC_init-Up2'; 'softStrSSC_init-Up2+'}
                    if (iter <= 1)
                        nu =1; 
                    else
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                    end
                    %% 
                    %C = ADMM_StrSSC(Xp, outliers, affine, alpha, Theta_old, gamma1, nu,  maxIter, Z);
                    [C, E] = ADMM_softStrSSC_Up(Xp, outliers, affine, alpha, softTheta, gamma1, nu,  maxIter, Z);                    
                    
                case {'softStrSSC_init2'; 'softStrSSC_init2+';}
                    if (iter <= 1)
                        nu =1; 
                    else
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                    end
                    %%
                    [C, E] = ADMM_softStrSSC(Xp, outliers, affine, alpha, softTheta, gamma1, nu,  maxIter, Z); 
                    
            end
            
            % relative changes in C
            tmp = Z - C;
            eval_iter(9, iter) = sum(abs(tmp(:))) / (sum(abs(C(:))) + (sum(abs(C(:))) < 1e-6));
            %% Initialize Z with the last optimal Z.
            Z = C;
            
            if (DEBUG)
                figure;
                image(abs(C)*800);colorbar;
            end

            if (affine==2)  % affine==2 : using L1 normalization. If rho <1 : filtering coefficients
                CKSym = BuildAdjacency(thrC(C,rho)); % L1 normalization + filtering
            else if (rho <1.0)
                    CKSym = thrC(C,rho); % e.g.,rho=0.7, it is filtering coefficients without L1 normalization
                    CKSym = 0.5* (abs(CKSym) + abs(CKSym'));
                else
                    CKSym = 0.5* (abs(C) + abs(C')); % no any postprocessing on C
                end
            end
            
            if (DEBUG)
                figure;
                image(abs(CKSym)*800);colorbar;
            end

            switch opt.sc_method                    
                case {'StrSSC_init2';'StrSSC_init-Fix2';'CStrSSC_init-Up2';...
                        'softStrSSC_init2';'softStrSSC_init-Fix2'; 'softStrSSC_init-Up2';}
                    [grps, ~, ~, Q, dim_kerN, num_spaU, sumD] = mSpectralClustering(CKSym, nbcluster);%grps           

                case {'StrSSC_init2+';'StrSSC_init-Fix2+';'CStrSSC_init-Up2+';...
                        'softStrSSC_init2+';'softStrSSC_init-Fix2+'; 'softStrSSC_init-Up2+';}
                    [grps, ~, ~, Q, dim_kerN, num_spaU, sumD] = mSpectralClustering(CKSym, nbcluster, grps);%grps           
            end              
                
            if (nbcluster >10)
                %missrate = 1 - compacc(grps, idx');
                missrate = 1 - evalAccuracyHungarian(grps, idx');
            else
                missrate = Misclassification(grps, idx');
            end
            
            av_conn = evalAverConn(0.5*(abs(C)+abs(C')), idx');
            [ssr_err, ssr ]  = evalSSR_error(C, idx');  
                
            softTheta = L2_distance(Q', Q');
            softTheta = (softTheta.^2) / 2;                
            %Theta = squareform(pdist(Q) .^2) / 2;
                
            tmp = softTheta - softTheta_old;  
            eval_iter(11, iter) = sum( abs(tmp(:)) ) / sum( abs(softTheta(:)));
            softTheta_old =softTheta;
            
            eval_iter(12, iter) =min(sumD(:)); % / (size(C, 1) * nbcluster);
            eval_iter(13, iter) =mean(sumD(:)); % / (size(C, 1) * nbcluster);            
            
            Theta = form_structure_matrix(grps);
                
            %affine = opt.affine;
            %outliers =opt.outliers;
            lambda0 = alpha / norm(Xp,1);
            if max(strcmp(opt.sc_method, {'StrSSC_init2';'StrSSC_init-Fix2';'CStrSSC_init-Up2';...
                    'StrSSC_init2+';'StrSSC_init-Fix2+';'CStrSSC_init-Up2+';}))                
                Z_Q_norm = sum(sum(abs((1-Theta).*C)));
                if (outliers)
                    obj = sum(sum(abs((1 + gamma0*(1-Theta)).*C))) + lambda0*sum(sum(E)); 
                else
                    obj = sum(sum(abs((1 + gamma0*(1-Theta)).*C))) + 0.5*lambda0* trace((Xp - Xp*C)*(Xp-Xp*C)');
                end
            end
            
            if max(strcmp(opt.sc_method, {'softStrSSC_init2';'softStrSSC_init-Fix2'; 'softStrSSC_init-Up2';...
                        'softStrSSC_init2+';'softStrSSC_init-Fix2+'; 'softStrSSC_init-Up2+';}))
                    %Z_Q_norm = sum(sum(abs((1-Theta).*C)));                    
                    Z_Q_norm = sum(sum(abs((softTheta).*C)));
                if (outliers)
                    obj = sum(sum(abs((1 + gamma0 * softTheta).*C))) + lambda0*sum(sum(E)); 
                else
                    obj = sum(sum(abs((1 + gamma0 * softTheta).*C))) + 0.5*lambda0* trace((Xp - Xp*C)*(Xp-Xp*C)');
                end
            end
            
            %% Relative Objective:
            if (iter ==1)
                relative_obj =1;
                eval_iter(8, iter) = relative_obj;
            else
                relative_obj =(obj - old_obj) /(old_obj + (old_obj < eps));
                eval_iter(8, iter) = relative_obj;
            end                              
  
            if (0==1)
                %figure;imagesc(softTheta, [0, 1]);title('soft \Theta'); colorbar;
                figure;imagesc(1-Theta, [0, 1]);colorbar;title('hard \Theta');
            end
            if (DEBUG)
                figure;
                image(abs(1-Theta)*800);colorbar;
            end
            
            %Z_Q_norm = sum(sum(abs((1-Theta).*C)));
            %disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);
            disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);                
            disp(['iter = ',num2str(iter),', rel_Z = ', num2str(eval_iter(9, iter)), ', rel_Theta =',num2str(eval_iter(10, iter)),',  rel_softTheta=', num2str(eval_iter(11, iter)), ', sumD = ', num2str(eval_iter(12, iter)),  ', mean sumD = ', num2str(eval_iter(13, iter))]);
                       
            eval_iter(1, iter) = 1 - missrate;
            eval_iter(2, iter) = av_conn;                
            eval_iter(3, iter) = ssr_err;
            %eval_iter(4, iter) = evalSSR_perc(C, idx');                
            eval_iter(4, iter) = obj;            
            eval_iter(5, iter) = Z_Q_norm;       

            eval_iter(6, iter) = dim_kerN;                
            eval_iter(7, iter) = num_spaU;             
            
            %%%%%%%%%%%%%%%%%%                         
            %% Stopping rules:
            m_Z_Q_norm =eval_iter(5, iter);
            m_rel_Theta =eval_iter(10, iter);
            
            if (num_spaU >0.9) || (dim_kerN >= nbcluster)                  
                odd_iter = odd_iter +1;
                if (m_rel_Theta < epsilon) || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon)) || (odd_iter >3)
                    break; 
                end
            else
                if (m_rel_Theta < epsilon) || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon))
                    break; % if Q didn't change, stop the iterations.
                end
            end
            acc_i =1 - missrate;            

%%           %%%%%%%%%%%%%%%%% An alternative stopping rule:            
%             m_rel_C =eval_iter(9, iter);
%             m_rel_softTheta =eval_iter(11, iter);
%             m_cost_kmeans =eval_iter(13, iter);            
%             if (num_spaU > 0)   
%                 acc_i =1 - missrate;
%                 break;                
%             else
%                 % if (m_rel_Theta <1e-8) || (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                 if (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || ...
%                         (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                     break;
%                 end
%             end
%             acc_i =1 - missrate;
%             m_rel_C_old = m_rel_C;
%             m_rel_softTheta_old = m_rel_softTheta;
%             m_cost_kmeans_old = m_cost_kmeans;            
%             %%%%%%%%%%%%%%%%%            

            old_obj = obj;
            Theta_old =Theta;
            m_Z_Q_norm_old = m_Z_Q_norm;
        end         
        
        %% Constraint hard S3C
    case {'hardCStrSSC2';'hardCStrSSC-Fix2';'hardCStrSSC-Up2';}
			
        gamma0 =opt.gamma0;
        opt.Z =zeros(size(D,2));
        alpha =opt.lambda;        
        r = opt.r; % r: the dimension of the target space when applying PCA or random projection
        Xp = DataProjection(D,r);
        %Theta_old =ones(size(X, 2)); % if we not initialize gamma1=0, we should initialize Theta_old =ones.    
        Theta_old =ones(size(D, 2));        
        %softTheta =zeros(size(D, 2));
        softTheta_old =1;
        iter_max =opt.iter_max;%iter_max =10; for SSC and Spectral Clustering
        maxIter =opt.maxIter; % in ADMM
        nu =opt.nu; % nu=1;
        rho =opt.SSCrho; %rho = 0.7;
        if isfield(opt,'C')
            Z = opt.C;
        else
            Z =zeros(size(Xp,2));
        end
        
        %% given prior side-information        
        switch opt.sc_method
           
            case {'hardCStrSSC-Fix2'; 'hardCStrSSC-Fix2+';'hardCStrSSC2'; 'hardCStrSSC2+'; 'hardCStrSSC-Up2'; 'hardCStrSSC-Up2+';}
      
                % use the given side-information
                Psi =opt.Psi;

        end
        
        eval_iter =zeros(16, iter_max);      
        old_obj =Inf;        
        m_cost_kmeans_old =Inf;
        %m_min_dist_old =1e-5;
        m_Z_Q_norm_old = Inf;       
        CONS_old = -1;

        iter =0; odd_iter =0;
        while (iter < iter_max)
            
            iter = iter +1;            
            switch opt.sc_method
                
                case {'hardCStrSSC2'; 'hardCStrSSC2+';}
                    if (iter <= 1)
                        %% This is the standard SSC with initialization of side-information
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC 
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_CStrSSC(Xp, outliers, affine, alpha, 1-Theta_old, Psi, gamma1, nu,  maxIter, Z);      
                    
                case  {'hardCStrSSC-Fix2'; 'hardCStrSSC-Fix2+'; }
                    if (iter <= 1)
                        %% This is the standard SSC with initialization of side-information
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC 
                        nu = nu * 1.0;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_CStrSSC(Xp, outliers, affine, alpha, 1-Theta_old, Psi, gamma1, nu,  maxIter, Z);                
                    
                case  {'hardCStrSSC-Up2'; 'hardCStrSSC-Up2+'; }
                    if (iter <= 1)
                        nu =1; gamma1 =0;
                    else
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_CStrSSC_Up(Xp, outliers, affine, alpha, 1-Theta_old, Psi, gamma1, nu,  maxIter, Z);                    
                    
            end                            
            
            % relative changes in C
            tmp = Z - C;
            eval_iter(9, iter) = sum(abs(tmp(:))) / (sum(abs(C(:))) + (sum(abs(C(:))) < 1e-6));
            %% Initialize Z with the last optimal Z.
            Z = C;
            
            if (DEBUG)
                figure;
                image(abs(C)*800);colorbar;
            end
            
            if (affine==2)  % affine==2 : using L1 normalization. If rho <1 : filtering coefficients
                CKSym = BuildAdjacency(thrC(C,rho)); % L1 normalization + filtering
            else if (rho <1.0)
                    CKSym = thrC(C,rho); % e.g.,rho=0.7, it is filtering coefficients without L1 normalization
                    CKSym = 0.5* (abs(CKSym) + abs(CKSym'));
                else
                    CKSym = 0.5* (abs(C) + abs(C')); % no any postprocessing on C
                    %CKSym = 0.5* (abs(C) + abs(C') + exp(Psi).*(Psi < 0) + exp(Psi').*(Psi' < 0)); % no any postprocessing on C
                end
            end
            
            if (DEBUG)
                figure;
                image(abs(CKSym)*800);colorbar;
            end
            
            %% Enhancement with MUST-LINK info:
            % t=4;
%             Psi_must =double(Psi <-0.5);
%             Psi_must_fused =double((Psi_must^t) > 0.5);
%             CKSym = CKSym + 0.1*Psi_must_fused;
%             t = t+1;
            
            switch opt.sc_method                    
                case {'hardCStrSSC2';'hardCStrSSC-Fix2';'hardCStrSSC-Up2'; }
                    [grps, ~, ~, Q, dim_kerN, num_spaU, sumD] = mSpectralClustering(CKSym, nbcluster, grps);%grps  
					
            end            

            if (nbcluster >10)
                missrate = 1 - evalAccuracyHungarian(grps, idx');
            else
                missrate = Misclassification(grps, idx');
            end                               

            av_conn = evalAverConn(0.5*(abs(C)+abs(C')), idx'); 
            [ssr_err, ssr ]  = evalSSR_error(C, idx');  

            softTheta = L2_distance(Q', Q');
            softTheta = (softTheta.^2) / 2;                
            %Theta = squareform(pdist(Q) .^2) / 2;

            tmp = softTheta - softTheta_old;  
            eval_iter(11, iter) = sum( abs(tmp(:)) ) / sum( abs(softTheta(:)));
            softTheta_old =softTheta;            

            eval_iter(12, iter) =min(sumD(:)); % / (size(C, 1) * nbcluster);
            eval_iter(13, iter) =mean(sumD(:)); % / (size(C, 1) * nbcluster);
            
            Theta = form_structure_matrix(grps);

            %affine = opt.affine;
            %outliers =opt.outliers;
            if (outliers)
                lambda0 = alpha / norm(Xp,1);
                obj = sum(sum(abs((Psi + gamma0*(1-Theta)).*C))) + lambda0*sum(sum(E)); 
            else
                %%                     
                lambda0 = alpha / norm(Xp,1);
                obj = sum(sum(abs((Psi + gamma0*(1-Theta)).*C))) + 0.5*lambda0* trace((Xp - Xp*C)*(Xp-Xp*C)'); 
            end
            
            %% Relative Objective:
            if (iter ==1)
                relative_obj =1;
                eval_iter(8, iter) = relative_obj;
            else
                relative_obj =(obj - old_obj) /(old_obj + (old_obj < eps));
                eval_iter(8, iter) = relative_obj;
            end                

            if (0==1)
                %figure;imagesc(softTheta, [0, 1]);title('soft \Theta'); colorbar;
                figure;imagesc(1-Theta, [0, 1]);colorbar;title('hard \Theta');
            end
            if (DEBUG)
                figure;
                image(abs(1-Theta)*800);colorbar;
            end
            
            Z_Q_norm = sum(sum(abs((1-Theta).*C)));
            
            %% Check the consistency CONS between Psi and Theta
            tmp = (Psi < -0.5) .* Theta;
            CONS_must = sum(tmp(:)) / (sum(sum((Psi< - 0.5))) + (sum(sum((Psi< - 0.5))) <1e-5));        
            if (sum(sum((Psi< - 0.5))) <1e-5)
                CONS_must =1;
            end
            
            tmp = (Psi > 0.5) .* (1 - Theta);
            CONS_cannot = sum(tmp(:)) / (sum(sum((Psi > 0.5))) + (sum(sum((Psi > 0.5))) <1e-5));         
            if (sum(sum((Psi > 0.5))) <1e-5)
                CONS_cannot =1;
            end
            
            CONS =(CONS_must + CONS_cannot)/2;            
            
            eval_iter(14, iter) =CONS_must; % CONS of within-cluster
            eval_iter(15, iter) =CONS_cannot; % CONS of between-cluster
            eval_iter(16, iter) =CONS; 
            
            %% Checking stop criterion
            tmp =Theta - Theta_old;            
            eval_iter(10, iter) = sum( abs(tmp(:)) ) / sum( abs(Theta(:)));
            %disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);

            disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);                
            disp(['           rel_Z = ', num2str(eval_iter(9, iter)), ', rel_Theta =',num2str(eval_iter(10, iter)),',  rel_softTheta=', num2str(eval_iter(11, iter)), ', sumD = ', num2str(eval_iter(12, iter)),  ', mean sumD = ', num2str(eval_iter(13, iter))]);
            disp(['           CONS_must = ', num2str(eval_iter(14, iter)), ', CONS_cannot =',num2str(eval_iter(15, iter)),',  CONS =', num2str(eval_iter(16, iter))]);

            eval_iter(1, iter) = 1 - missrate;
            eval_iter(2, iter) = av_conn;                
            eval_iter(3, iter) = ssr_err;
            %eval_iter(4, iter) = evalSSR_perc(C, idx');                
            eval_iter(4, iter) = obj;            
            eval_iter(5, iter) = Z_Q_norm;       

            eval_iter(6, iter) = dim_kerN;                
            eval_iter(7, iter) = num_spaU;             
            %%%%%%%%%%%%%%%%%%                  

            %% Stopping rules:
            m_Z_Q_norm =eval_iter(5, iter);
            m_rel_Theta =eval_iter(10, iter);
            m_cost_kmeans =eval_iter(13, iter); 
            
             if (num_spaU >0.9) || (dim_kerN >= nbcluster)                  
                odd_iter = odd_iter +1; % || (odd_iter >3)
                %if (m_rel_Theta < epsilon) || abs(relative_obj) <5e-2 || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon))  || (m_cost_kmeans - m_cost_kmeans_old < m_min_dist_old)
                if (m_rel_Theta < epsilon) || abs(relative_obj) <5e-3 || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon))  || (m_cost_kmeans_old - m_cost_kmeans < 5e-3) || (CONS - CONS_old < epsilon)
                    if (CONS > CONS_old)
                        acc_i =1 - missrate;
                    end                          
                    break; 
                end
             else %  || (odd_iter >3)
                %if (m_rel_Theta < epsilon) || abs(relative_obj) <5e-2 || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon)) || (m_cost_kmeans - m_cost_kmeans_old < m_min_dist_old)
                if (m_rel_Theta < epsilon) || abs(relative_obj) <5e-3 || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon))  || (m_cost_kmeans_old - m_cost_kmeans < 5e-3) || (CONS - CONS_old < epsilon)
                    if (CONS > CONS_old)
                        acc_i =1 - missrate;
                    end                          
                    break; % if Q didn't change, stop the iterations.
                end
            end
            acc_i =1 - missrate;
            
%%           %%%%%%%%%%%%%%%%% An alternative stopping rule:            
%             m_rel_C =eval_iter(9, iter);
%             m_rel_softTheta =eval_iter(11, iter);
%             m_cost_kmeans =eval_iter(13, iter);            
%             if (num_spaU > 0)   
%                 acc_i =1 - missrate;
%                 break;                
%             else
%                 % if (m_rel_Theta <1e-8) || (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                 if (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || ...
%                         (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                     break;
%                 end
%             end
%             acc_i =1 - missrate;
%             m_rel_C_old = m_rel_C;
%             m_rel_softTheta_old = m_rel_softTheta;
%             m_cost_kmeans_old = m_cost_kmeans;            
%             %%%%%%%%%%%%%%%%%
            
            old_obj = obj;
            Theta_old =Theta;
            CONS_old =CONS;
            m_Z_Q_norm_old = m_Z_Q_norm;        
            m_cost_kmeans_old = m_cost_kmeans;                 
            %m_min_dist_old =eval_iter(12, iter);
        end             
        
    case {'softCStrSSC2';'softCStrSSC-Fix2'; 'softCStrSSC-Up2';} 
        gamma0 =opt.gamma0;
        opt.Z =zeros(size(D,2));
        alpha =opt.lambda;        
        r = opt.r; % r: the dimension of the target space when applying PCA or random projection
        Xp = DataProjection(D,r);
        %Theta_old =ones(size(X, 2)); % if we not initialize gamma1=0, we should initialize Theta_old =ones.       
        Theta_old =ones(size(D, 2));
        softTheta =zeros(size(D, 2));
        softTheta_old =1;
        iter_max =opt.iter_max;%iter_max =10; for SSC and Spectral Clustering
        maxIter =opt.maxIter; % in ADMM
        nu =opt.nu; % nu=1;
        rho =opt.SSCrho; %rho = 0.7;
        Z =zeros(size(Xp,2));
        
        %% given prior side-information        
        switch opt.sc_method
           
            case {'softCStrSSC-Fix2'; 'softCStrSSC-Fix2+';'softCStrSSC2'; 'softCStrSSC2+'; 'softCStrSSC-Up2'; 'softCStrSSC-Up2+';}
      
                % use the given side-information
                Psi =opt.Psi;        

        end
        
        eval_iter =zeros(16, iter_max);      
        old_obj =Inf;
        m_cost_kmeans_old =Inf;
        %m_min_dist_old =1e-5;
        m_Z_Q_norm_old = Inf;
        CONS_old = -1;
                
        iter =0; odd_iter =0;
        while (iter < iter_max)            
            iter = iter +1;            
            switch opt.sc_method                
                %% 
                case {'softCStrSSC-Fix2'; 'softCStrSSC-Fix2+'; }
                    if (iter <= 1)
                        %% This is the standard SSC with initialization of side-info
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC 
                        nu = nu * 1.0;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_CStrSSC(Xp, outliers, affine, alpha, softTheta, Psi, gamma1, nu,  maxIter, Z);             
                    
                 %%
                case {'softCStrSSC-Up2'; 'softCStrSSC-Up2+'; }
                    if (iter <= 1)
                        %% This is the standard SSC with initialization of side-info
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %% 
                    [C, E] = ADMM_CStrSSC_Up(Xp, outliers, affine, alpha, softTheta, Psi, gamma1, nu,  maxIter, Z);                    
                    
                case {'softCStrSSC2'; 'softCStrSSC2+'; }
                    if (iter <= 1)
                        %% This is the standard SSC with initialization of side-info
                        nu =1; gamma1 =0;
                    else
                        %% This is the re-weighted SSC 
                        nu = nu * 1.2;%1.1, 1.2;%1.5, 1.2
                        gamma1 =gamma0;
                    end
                    %%
                    [C, E] = ADMM_CStrSSC(Xp, outliers, affine, alpha, softTheta, Psi, gamma1, nu,  maxIter, Z);
            end                
            
            % relative changes in C
            tmp = Z - C;
            eval_iter(9, iter) = sum(abs(tmp(:))) / (sum(abs(C(:))) + (sum(abs(C(:))) < 1e-6));
                        
            %% Initialize Z with the last optimal Z.
            Z =C;      
            
            if (DEBUG)
                figure;
                colormap hot;
                image(abs(C)*800);colorbar;
            end
            
            if (affine==2)  % affine==2 : using L1 normalization. If rho <1 : filtering coefficients
                CKSym = BuildAdjacency(thrC(C,rho)); % L1 normalization + filtering
            else if (rho <1.0)
                    CKSym = thrC(C,rho); % e.g.,rho=0.7, it is used to filter the coefficients (without L1 normalization)
                    CKSym = 0.5* (abs(CKSym) + abs(CKSym'));
                else
                    CKSym = 0.5* (abs(C) + abs(C')); % if affine=0 or 1, and rho=1, then there is no any postprocessing on C
                end
            end            
            
            %% Enhancement with MUST-LINK info:
            % t=4;
%             Psi_must =double(Psi <-0.5);
%             Psi_must_fused =double((Psi_must^t) > 0.5);
%             CKSym = CKSym + 0.1*Psi_must_fused;
%             t =t+1;

            if (DEBUG)
                figure;
                image(abs(CKSym)*800);colorbar;
            end

            switch opt.sc_method                    
                case {'softCStrSSC2';'softCStrSSC-Fix2';'softCStrSSC-Up2';}
                    [grps, ~, ~, Q, dim_kerN, num_spaU, sumD] = mSpectralClustering(CKSym, nbcluster, grps);%grps           
            
            end                  
            softTheta = L2_distance(Q', Q');
            softTheta = (softTheta.^2) / 2;
            %Theta = squareform(pdist(Q) .^2) / 2;        

            Theta = form_structure_matrix(grps);
            
            %% Check the consistency CONS between Psi and Theta
            tmp = (Psi < -0.5) .* Theta;
            CONS_must = sum(tmp(:)) /sum(sum((Psi< - 0.5)));        
            
            tmp = (Psi > 0.5) .* (1 - Theta);
            CONS_cannot = sum(tmp(:)) /sum(sum((Psi > 0.5)));            
            
            CONS =(CONS_must + CONS_cannot)/2;            
            
            eval_iter(14, iter) =CONS_must; % CONS of within-cluster
            eval_iter(15, iter) =CONS_cannot; % CONS of between-cluster
            eval_iter(16, iter) =CONS; 

            %affine = opt.affine;
            %outliers =opt.outliers;
            if (outliers)
                lambda0 = alpha / norm(Xp,1);
                obj = sum(sum(abs((Psi + gamma0*softTheta).*C))) + lambda0*sum(sum(E)); 
            else
                %%                     
                lambda0 = alpha / norm(Xp,1);
                obj = sum(sum(abs((Psi + gamma0*softTheta).*C))) + 0.5*lambda0* trace((Xp - Xp*C)*(Xp-Xp*C)'); 
            end
                        
            if (nbcluster >10)
                %missrate = 1 - compacc(grps, idx');
                missrate = 1 - evalAccuracyHungarian(grps, idx');
            else
                missrate = Misclassification(grps, idx');
            end                               
            
            av_conn = evalAverConn(0.5*(abs(C)+abs(C')), idx'); 
            [ssr_err, ssr ]  = evalSSR_error(C, idx');  
            
            eval_iter(1, iter) = 1 - missrate;
            eval_iter(2, iter) = av_conn; 
            eval_iter(3, iter) = ssr_err;
            %eval_iter(4, iter) = evalSSR_perc(C, idx');                
            eval_iter(4, iter) = obj;            
            
            Z_Q_norm = sum(sum(abs((softTheta).*C)));
            
            eval_iter(5, iter) = Z_Q_norm;       
            eval_iter(6, iter) = dim_kerN;                
            eval_iter(7, iter) = num_spaU;                 
            
            %% Relative Objective:
            if (iter ==1)
                acc_i =1 - missrate;
                relative_obj =1;
                eval_iter(8, iter) = relative_obj;
            else
                relative_obj =(obj - old_obj) /(old_obj + (old_obj < eps));
                eval_iter(8, iter) = relative_obj;
            end                
                
            if (0==1)
                figure;imagesc(softTheta, [0, 1]);title('soft \Theta'); colorbar;
                figure;imagesc(1-Theta, [0, 1]);colorbar;title('hard \Theta');
            end
            if (DEBUG)
                figure;colormap hot
                image(abs(1-Theta)*800);colorbar;                
                figure;colormap hot
                imagesc(softTheta, [0, 1]);title('soft \Theta'); colorbar;
                figure;colormap hot
                image(softTheta*500);title('soft \Theta'); colorbar;
            end
 
            %disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);
            
            %% Checking stopping condition
            tmp = softTheta - softTheta_old;
            eval_iter(11, iter) = sum( abs(tmp(:)) ) / sum( abs(softTheta(:)));
            softTheta_old =softTheta;
            
            tmp =Theta - Theta_old;            
            eval_iter(10, iter) = sum( abs(tmp(:)) ) / sum( abs(Theta(:)));

            eval_iter(12, iter) =min(sumD(:)); % / (size(C, 1) * nbcluster);
            eval_iter(13, iter) =mean(sumD(:)); % / (size(C, 1) * nbcluster);                        
            
            disp(['iter = ',num2str(iter),', acc = ', num2str(1 - missrate), ', rel_obj. =',num2str(relative_obj),',  obj. =', num2str(obj), ', Z_Q_norm = ', num2str(Z_Q_norm)]);                
            disp(['           rel_Z = ', num2str(eval_iter(9, iter)), ', rel_Theta =',num2str(eval_iter(10, iter)),',  rel_softTheta=', num2str(eval_iter(11, iter)), ', sumD = ', num2str(eval_iter(12, iter)),  ', mean sumD = ', num2str(eval_iter(13, iter))]);
            disp(['           CONS_must = ', num2str(eval_iter(14, iter)), ', CONS_cannot =',num2str(eval_iter(15, iter)),',  CONS =', num2str(eval_iter(16, iter))]);
                     
            %% Stopping rules:
            m_Z_Q_norm =eval_iter(5, iter);
            m_rel_Theta =eval_iter(10, iter);
            m_cost_kmeans =eval_iter(13, iter);   
            
            if (num_spaU >0.9) || (dim_kerN >= nbcluster)                  
                odd_iter = odd_iter +1; %  || (odd_iter >3)
                %if (m_rel_Theta < epsilon) || abs(relative_obj) <5e-2 || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon))  || (m_cost_kmeans - m_cost_kmeans_old < m_min_dist_old)               
                if (m_rel_Theta < epsilon) || abs(relative_obj) <5e-3 || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon)) || (m_cost_kmeans_old - m_cost_kmeans <  5e-3) || (CONS - CONS_old < epsilon)
                    if (CONS > CONS_old)
                        acc_i =1 - missrate;
                    end               
                    break; 
                end
            else %  || (odd_iter >3)
                %if (m_rel_Theta < epsilon) || abs(relative_obj) <5e-2 || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon)) || (m_cost_kmeans - m_cost_kmeans_old < m_min_dist_old)
                if (m_rel_Theta < epsilon) || abs(relative_obj) <5e-3 || ((m_Z_Q_norm_old - m_Z_Q_norm < epsilon) && (old_obj - obj < epsilon)) || (m_cost_kmeans_old - m_cost_kmeans < 5e-3) || (CONS - CONS_old < epsilon)
                    if (CONS > CONS_old)
                        acc_i =1 - missrate;
                    end
                    break; % if Q didn't change, stop the iterations.
                end
            end
            
            acc_i =1 - missrate;
            
%%           %%%%%%%%%%%%%%%%% An alternative stopping rule:            
%             m_rel_C =eval_iter(9, iter);
%             m_rel_softTheta =eval_iter(11, iter);
%             m_cost_kmeans =eval_iter(13, iter);            
%             if (num_spaU > 0)   
%                 acc_i =1 - missrate;
%                 break;                
%             else
%                 % if (m_rel_Theta <1e-8) || (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                 if (m_rel_softTheta_old - m_rel_softTheta < 1e-8 ) || (m_Z_Q_norm_old - m_Z_Q_norm < 1e-8) || ...
%                         (m_rel_C_old - m_rel_C < 1e-8) || (m_cost_kmeans_old - m_cost_kmeans < 1e-8)
%                     break;
%                 end
%             end
%             acc_i =1 - missrate;
%             m_rel_C_old = m_rel_C;
%             m_rel_softTheta_old = m_rel_softTheta;
%             m_cost_kmeans_old = m_cost_kmeans;            
%             %%%%%%%%%%%%%%%%%

            old_obj = obj;
            Theta_old =Theta;
            CONS_old = CONS;
            m_Z_Q_norm_old = m_Z_Q_norm;            
            m_cost_kmeans_old = m_cost_kmeans;                 
            %m_min_dist_old =eval_iter(12, iter);
        end             
                
    otherwise        
        disp('Unkown Subspace Clustering Methods...');        
end

function M = form_structure_matrix(idx,n)
if nargin<2
    n =size(idx,2);
end
M =zeros(n);
id =unique(idx);
for i =1:length(id)
    idx_i =find(idx == id(i));
    M(idx_i,idx_i)=ones(size(idx_i,2));
end  

% function Dist = form_distance_matrix(X,t)
% if nargin<2
%     t =1;
% end
% 
% for i =1:length(id)
%     idx_i =find(idx == id(i));
%     M(idx_i,idx_i)=ones(size(idx_i,2));
% end  